<?php
/**
 * Plugin Name: WooCommerce Telegram уведомления
 * Description: Отправляет уведомления о заказах WooCommerce в Telegram.
 * Version: 2.8
 * Author: revtail.ru
 * Author URI: https://revtail.ru/
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!function_exists('custom_plugin_update_info_wtn')) {
	function custom_plugin_update_info_wtn($transient) {
		if (empty($transient->checked)) {
			return $transient;
		}

		// Уникальный slug плагина
		$plugin_slug = 'woocommerce-telegram-notify';

		// Запрос к серверу обновлений
		$response = wp_remote_get("https://revtail.ru/wordpress-update-plugins/update-server.php?plugin=$plugin_slug");

		if (is_wp_error($response)) {
			return $transient;
		}

		$data = json_decode(wp_remote_retrieve_body($response));
		if (!$data || isset($data->error)) {
			return $transient;
		}

		// Текущая версия
		$current_version = '2.8';

		// Добавляем обновление, если оно доступно
		if (version_compare($current_version, $data->new_version, '<')) {
			$transient->response = array_merge($transient->response ?? [], [
				"$plugin_slug/$plugin_slug.php" => (object) [
					'slug'        => $plugin_slug,
					'new_version' => $data->new_version,
					'package'     => $data->package,
					'url'         => $data->url,
				]
			]);
		}

		return $transient;
	}
}

// Подключаем к системе обновлений WordPress
add_filter('site_transient_update_plugins', 'custom_plugin_update_info_wtn', 10, 1);

if (!function_exists('custom_plugin_update_details')) {
	function custom_plugin_update_details($false, $action, $args) {
		if ($action !== 'plugin_information' || $args->slug !== 'woocommerce-telegram-notify') {
			return $false;
		}

		$response = wp_remote_get("https://revtail.ru/wordpress-update-plugins/update-server.php?plugin=$plugin_slug");
		
		if (is_wp_error($response)) {
			return $false;
		}

		$data = json_decode(wp_remote_retrieve_body($response));

		if (!$data) {
			return $false;
		}

		return (object) [
			'name'           => 'WooCommerce Telegram уведомления',
			'slug'           => 'woocommerce-telegram-notify',
			'version'        => $data->new_version,
			'author'         => 'revtail',
			'homepage'       => 'https://revtail.ru/',
			'download_link'  => $data->package,
			'sections'       => [
				'description'  => 'Описание вашего плагина.',
				'changelog'    => 'Список изменений в новой версии.',
			],
		];
	}
}

// Добавляем хук для вывода информации об обновлении
add_filter('plugins_api', 'custom_plugin_update_details', 10, 3);

// Добавляем страницу настроек
function wctn_add_settings_page() {
    add_options_page(
        'Настройки Telegram Notify',
        'Telegram Notify',
        'manage_options',
        'wctn-settings',
        'wctn_settings_page'
    );
}
add_action('admin_menu', 'wctn_add_settings_page');

// Рендерим страницу настроек
function wctn_settings_page() {
    ?>
    <div class="wrap">
        <h1>Настройки Telegram Notify</h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('wctn_settings_group');
            do_settings_sections('wctn-settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

// Регистрируем настройки
function wctn_register_settings() {
    register_setting('wctn_settings_group', 'wctn_telegram_token');
    register_setting('wctn_settings_group', 'wctn_telegram_chat_id');
    register_setting('wctn_settings_group', 'wctn_send_after_payment');

    add_settings_section('wctn_main_section', 'Основные настройки', null, 'wctn-settings');

    add_settings_field(
        'wctn_telegram_token', 'API-токен Telegram бота',
        'wctn_telegram_token_callback', 'wctn-settings', 'wctn_main_section'
    );

    add_settings_field(
        'wctn_telegram_chat_id', 'ID группы Telegram',
        'wctn_telegram_chat_id_callback', 'wctn-settings', 'wctn_main_section'
    );

    add_settings_field(
        'wctn_send_after_payment', 'Отправлять уведомление только после оплаты',
        'wctn_send_after_payment_callback', 'wctn-settings', 'wctn_main_section'
    );
}
add_action('admin_init', 'wctn_register_settings');

function wctn_telegram_token_callback() {
    $value = get_option('wctn_telegram_token', '');
    echo "<input type='text' name='wctn_telegram_token' value='" . esc_attr($value) . "' style='width: 400px;'>";
}

function wctn_telegram_chat_id_callback() {
    $value = get_option('wctn_telegram_chat_id', '');
    echo "<input type='text' name='wctn_telegram_chat_id' value='" . esc_attr($value) . "' style='width: 400px;'>";
}

function wctn_send_after_payment_callback() {
    $value = get_option('wctn_send_after_payment', 'no');
    echo "<input type='checkbox' name='wctn_send_after_payment' value='yes' " . checked($value, 'yes', false) . ">";
}

// Функция отправки сообщений в Telegram
function wctn_send_telegram_message($message) {
    $token = get_option('wctn_telegram_token');
    $chat_id = get_option('wctn_telegram_chat_id');
    
    if (!$token || !$chat_id) return;
    
    $url = "https://api.telegram.org/bot$token/sendMessage";
    
    $data = [
        'chat_id' => $chat_id,
        'text' => $message,
        'parse_mode' => 'Markdown'
    ];
    
    wp_remote_post($url, [
        'body' => $data,
        'timeout' => 10,
    ]);
}

// Сохраняем значения полей при оформлении заказа
function save_custom_checkout_fields_telegram_notify($order_id) {
    if (isset($_POST['billing_new_fild10'])) {
        update_post_meta($order_id, '_billing_new_fild10', sanitize_text_field($_POST['billing_new_fild10']));
    }
    if (isset($_POST['billing_new_fild11'])) {
        update_post_meta($order_id, '_billing_new_fild11', sanitize_text_field($_POST['billing_new_fild11']));
    }
}
add_action('woocommerce_checkout_update_order_meta', 'save_custom_checkout_fields');

// Хук на изменение статуса заказа
function wctn_notify_order_status_changed($order_id, $old_status, $new_status) {
    $send_after_payment = get_option('wctn_send_after_payment', 'no') === 'yes';
    
    if ($send_after_payment && !in_array($new_status, ['processing', 'completed'])) {
        return;
    }
    
    $order = wc_get_order($order_id);
    if (!$order) return;
    
    $customer_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
    $customer_phone = $order->get_billing_phone();
	$customer_email = $order->get_billing_email();
    $order_total = $order->get_total();
    $payment_method = $order->get_payment_method_title();
    $order_status = wc_get_order_status_name($new_status);
    $order_date = date_i18n('j F Y, H:i', strtotime($order->get_date_created()));
    $shipping_method = $order->get_shipping_method();
    $delivery_date = get_post_meta($order_id, 'dtwc_delivery_date', true);
    $delivery_time = get_post_meta($order_id, 'dtwc_delivery_time', true);
	$customer_last_name = get_post_meta($order_id, '_billing_new_fild10', true);
	$customer_whatsapp = get_post_meta($order_id, '_billing_new_fild11', true);
	$customer_whatsapp = preg_replace('/\D/', '', $customer_whatsapp);
    
    $items = "\n-------------  *ТОВАРЫ*  -------------\n";
    foreach ($order->get_items() as $item) {
        $items .= "*{$item->get_quantity()} x {$item->get_name()}* - {$item->get_total()} RUB\n";
    }
    $items .= "--------------------------------------\n";
	$order_comments = $order->get_customer_note();
    
	$billing_parts = array_filter([
		$order->get_billing_state(),
		$order->get_billing_city(),
		$order->get_billing_address_1(),
		$order->get_billing_address_2()
	]);
	$billing_address = implode(', ', $billing_parts);
	
    $shipping_parts = array_filter([
		$order->get_shipping_state(),
		$order->get_shipping_city(),
		$order->get_shipping_address_1(),
		$order->get_shipping_address_2()
	]);
	$shipping_address = implode(', ', $shipping_parts);
    
    $message = "*Новый заказ №$order_id*\n\n";
    $message .= "*Имя:* $customer_name\n";
	if (!empty($customer_last_name)) {
	$message .= "*Отчество:* $customer_last_name\n";
	}
    $message .= "*Сумма:* $order_total RUB\n";
    $message .= "*Оплата:* $payment_method\n";
    $message .= "*Статус заказа:* $order_status\n";
    $message .= "*Дата заказа:* $order_date\n";
	if (!empty($customer_email)) {
		$customer_email = str_replace('_', '\_', $customer_email);
		$message .= "*E-mail:* $customer_email\n";
	}
	if (!empty($customer_whatsapp)) {
	$message .= "*WhatsApp:* +$customer_whatsapp\n";
	}
    if (!empty($customer_phone)) {
    $message .= "*Телефон клиента:* +7$customer_phone\n\n";
	} else {
		$message .= "\n";
	}
    $message .= "*Способ доставки:* $shipping_method\n";
	if (!empty($delivery_date)) {
	$message .= "*Дата доставки:* $delivery_date\n";
	}
	if (!empty($delivery_time)) {
	$message .= "*Время доставки:* $delivery_time\n";
	}
    $message .= $items . "\n";
	if (!empty($billing_address)) {
    $message .= "*Примечание:* `$order_comments`\n\n";
	} else {
		$message .= "\n";
	}
    $message .= "*ПЛАТЕЛЬЩИК:*\n";
    $message .= "$customer_name\n";
	if (!empty($billing_address)) {
    $message .= "*Адрес:* `$billing_address`\n\n";
	} else {
		$message .= "\n";
	}
    $message .= "*ДОСТАВКА:*\n";
    $message .= "$customer_name\n";
		if (!empty($shipping_address)) {
	$message .= "*Адрес:* `$shipping_address`\n";
	}
    
    wctn_send_telegram_message($message, "Markdown");
}

add_action('woocommerce_order_status_changed', 'wctn_notify_order_status_changed', 10, 3);

function add_inputmask_script() {
    // Проверяем, что мы на странице оформления заказа WooCommerce
    if (is_checkout()) {
        // Подключаем библиотеку Inputmask для работы с масками
        wp_enqueue_script(
            'inputmask',
            'https://cdnjs.cloudflare.com/ajax/libs/jquery.inputmask/5.0.7/jquery.inputmask.min.js',
            array('jquery'),
            null,
            true
        );

        // Подключаем кастомный JS, который будет применять маску
        wp_enqueue_script(
            'custom-phone-mask',
            plugin_dir_url(__FILE__) . 'assets/js/custom.js',  // Путь к файлу custom.js
            array('jquery', 'inputmask'),
            null,
            true
        );
    }
}
add_action('wp_enqueue_scripts', 'add_inputmask_script');


// Обрабатываем номер телефона ДО сохранения данных в БД
add_action('woocommerce_checkout_create_order', 'preprocess_phone_before_save', 10, 2);

function preprocess_phone_before_save($order, $data) {
    // Получить введённый номер телефона
    $phone = sanitize_text_field(wc_clean($data['billing_phone']));

    // Привести номер к нужному виду (убрать посторонние символы)
    $clean_phone = preg_replace('/[^+\d]/', '', $phone);

    // Проверяем наличие "+7" и добавляем, если нужно
    if (substr($clean_phone, 0, 2) !== '+7') {
        $clean_phone = '+7' . ltrim($clean_phone, '+');
    }

    // Назначаем подготовленное значение обратно в объект заказа
    $order->set_billing_phone($clean_phone);
}

// Название директории и название основного файла плагина Woocommerce
$required_plugin = 'woocommerce/woocommerce.php';

// Проверяем активность плагина
if ( is_plugin_active($required_plugin) ) {
    // Если плагин установлен и активирован, продолжаем выполнение нашего плагина
    add_action('plugins_loaded', 'my_plugin_init');
    
    function my_plugin_init() {
        // Логика активации вашего плагина
        echo "Ваш плагин успешно активировался!";
    }
} else {
    // Сообщение об ошибке при отсутствии активного плагина
    deactivate_plugins(plugin_basename(__FILE__)); // Деактивируем свой плагин
    wp_die("Ошибка: ваш плагин требует активной версии плагина Woocommerce.");
}