<?php
/*
Plugin Name: WC Delivery Times
Description: Выбор даты и времени доставки с учётом рабочего времени и времени на приготовление.
Version: 1.4
Author: revtail.ru
*/

if (!function_exists('custom_plugin_update_info_wtn')) {
	function custom_plugin_update_info_wtn($transient) {
		if (empty($transient->checked)) {
			return $transient;
		}

		// Уникальный slug плагина
		$plugin_slug = 'woocommerce-delivery-times';

		// Запрос к серверу обновлений
		$response = wp_remote_get("https://revtail.ru/wordpress-update-plugins/update-server.php?plugin=$plugin_slug");

		if (is_wp_error($response)) {
			return $transient;
		}

		$data = json_decode(wp_remote_retrieve_body($response));
		if (!$data || isset($data->error)) {
			return $transient;
		}

		// Текущая версия
		$current_version = '1.4';

		// Добавляем обновление, если оно доступно
		if (version_compare($current_version, $data->new_version, '<')) {
			$transient->response = array_merge($transient->response ?? [], [
				"$plugin_slug/$plugin_slug.php" => (object) [
					'slug'        => $plugin_slug,
					'new_version' => $data->new_version,
					'package'     => $data->package,
					'url'         => $data->url,
				]
			]);
		}

		return $transient;
	}
}

// Подключаем к системе обновлений WordPress
add_filter('site_transient_update_plugins', 'custom_plugin_update_info_wtn', 10, 1);

if (!function_exists('custom_plugin_update_details')) {
	function custom_plugin_update_details($false, $action, $args) {
		if ($action !== 'plugin_information' || $args->slug !== 'woocommerce-delivery-times') {
			return $false;
		}

		$response = wp_remote_get("https://revtail.ru/wordpress-update-plugins/update-server.php?plugin=$plugin_slug");
		
		if (is_wp_error($response)) {
			return $false;
		}

		$data = json_decode(wp_remote_retrieve_body($response));

		if (!$data) {
			return $false;
		}

		return (object) [
			'name'           => 'WC Delivery Times',
			'slug'           => 'woocommerce-delivery-times',
			'version'        => $data->new_version,
			'author'         => 'revtail.ru',
			'homepage'       => 'https://revtail.ru/',
			'download_link'  => $data->package,
			'sections'       => [
				'description'  => 'Описание вашего плагина.',
				'changelog'    => 'Список изменений в новой версии.',
			],
		];
	}
}

// Добавляем хук для вывода информации об обновлении
add_filter('plugins_api', 'custom_plugin_update_details', 10, 3);

// Подключение Flatpickr
add_action('wp_enqueue_scripts', function() {
    if (is_checkout()) {
        wp_enqueue_script('flatpickr', 'https://cdn.jsdelivr.net/npm/flatpickr', [], null, true);
        wp_enqueue_script('flatpickr-locale-ru', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/l10n/ru.js', ['flatpickr'], null, true);
        wp_enqueue_style('flatpickr-style', 'https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css');
    }
});

// Получение настроек доставки
function get_delivery_settings() {
    $options = get_option('wcdtwc_delivery_times_settings');
    return [
        'working_hours' => $options['working_hours'] ?? '10:00-22:00',
        'preparation_time' => intval($options['preparation_time'] ?? 30),
        'required' => $options['delivery_datetime_required'] ?? 'no'
    ];
}

// Вывод полей даты и времени на странице оформления заказа
add_action('woocommerce_before_order_notes', function($checkout) {
    $settings = get_delivery_settings();
    list($start_time, $end_time) = explode('-', $settings['working_hours']);
    $is_required = $settings['required'] === 'yes';

    echo '<h3>Выберите дату и время доставки</h3>';

    woocommerce_form_field('delivery_date', [
        'type' => 'text',
        'class' => ['form-row-wide'],
        'label' => 'Дата доставки',
        'required' => $is_required
    ], $checkout->get_value('delivery_date'));

    echo '<div class="form-row form-row-wide delivery-time-wrapper" style="display:none;">';
    woocommerce_form_field('delivery_time', [
        'type' => 'select',
        'label' => 'Время доставки',
        'required' => $is_required,
        'options' => ['' => 'Сначала выберите дату']
    ], $checkout->get_value('delivery_time'));
    echo '</div>';
    ?>
    <script>
    jQuery(function($) {
        const prepTime = <?php echo intval($settings['preparation_time']); ?>;
        const workingStart = "<?php echo esc_js($start_time); ?>";
        const workingEnd = "<?php echo esc_js($end_time); ?>";

        function generateTimeOptions(selectedDate) {
			const now = new Date();
			const selected = new Date(selectedDate);
			selected.setHours(0, 0, 0, 0);
			now.setSeconds(0, 0);

			let startParts = workingStart.split(':');
			let endParts = workingEnd.split(':');
			let start = new Date(selectedDate);
			start.setHours(+startParts[0], +startParts[1], 0, 0);
			let end = new Date(selectedDate);
			end.setHours(+endParts[0], +endParts[1], 0, 0);

			// Только если выбрана сегодняшняя дата — добавляем время приготовления
			const today = new Date();
			today.setHours(0, 0, 0, 0);
			if (selected.getTime() === today.getTime()) {
				let earliest = new Date();
				earliest.setMinutes(earliest.getMinutes() + prepTime);
				let minutes = earliest.getMinutes();
				earliest.setMinutes(minutes + (30 - minutes % 30)); // округляем вверх до ближайших 30 минут
				earliest.setSeconds(0, 0);
				if (start < earliest) start = earliest;
			}

			let times = {};
			while (start < end) {
				const h = start.getHours().toString().padStart(2, '0');
				const m = start.getMinutes().toString().padStart(2, '0');
				const label = `${h}:${m}`;
				times[label] = label;
				start.setMinutes(start.getMinutes() + 30);
			}

			const $wrapper = $('.delivery-time-wrapper');
			const $timeField = $wrapper.find('select[name="delivery_time"]');
			$timeField.empty();
			$timeField.append($('<option>', { value: '', text: 'Выберите время' }));
			$.each(times, function(val, text) {
				$timeField.append($('<option>', { value: val, text: text }));
			});

			$wrapper.show();
		}

        flatpickr("input[name='delivery_date']", {
            dateFormat: "Y-m-d",
            minDate: "today",
            maxDate: new Date().fp_incr(7),
            locale: "ru",
            onChange: function(selectedDates, dateStr) {
                if (selectedDates.length > 0) {
                    generateTimeOptions(selectedDates[0]);
                }
            }
        });
    });
    </script>
    <?php
});

// Сохраняем мета-данные при оформлении заказа
add_action('woocommerce_checkout_update_order_meta', function($order_id) {
    if (!empty($_POST['delivery_date'])) {
        update_post_meta($order_id, 'dtwc_delivery_date', sanitize_text_field($_POST['delivery_date']));
        update_post_meta($order_id, 'dtwc_delivery_time', sanitize_text_field($_POST['delivery_time']));
    }
});

// Отображаем дату и время доставки в админке заказа
add_action('woocommerce_admin_order_data_after_billing_address', function($order) {
    $delivery_date = get_post_meta($order->get_id(), 'dtwc_delivery_date', true);
    $delivery_time = get_post_meta($order->get_id(), 'dtwc_delivery_time', true);
    if ($delivery_date) {
        echo '<p><strong>Дата доставки:</strong> ' . esc_html($delivery_date) . '</p>';
        echo '<p><strong>Время доставки:</strong> ' . esc_html($delivery_time) . '</p>';
    }
});

// Проверка обязательности при оформлении
add_action('woocommerce_checkout_process', function() {
    $settings = get_delivery_settings();
    if ($settings['required'] === 'yes') {
        if (empty($_POST['delivery_date'])) {
            wc_add_notice('Пожалуйста, выберите дату доставки.', 'error');
        }
        if (empty($_POST['delivery_time'])) {
            wc_add_notice('Пожалуйста, выберите время доставки.', 'error');
        }
    }
});

// Вывод даты и времени доставки в письмах
add_action('woocommerce_email_order_meta', function($order, $sent_to_admin, $plain_text, $email) {
    $delivery_date = get_post_meta($order->get_id(), 'dtwc_delivery_date', true);
    $delivery_time = get_post_meta($order->get_id(), 'dtwc_delivery_time', true);
    if ($delivery_date && $delivery_time) {
        if ($plain_text) {
            echo "Дата доставки: $delivery_date\n";
            echo "Время доставки: $delivery_time\n";
        } else {
            echo '<p><strong>Дата доставки:</strong> ' . esc_html($delivery_date) . '</p>';
            echo '<p><strong>Время доставки:</strong> ' . esc_html($delivery_time) . '</p>';
        }
    }
}, 20, 4);

// Страница настроек в админке
add_action('admin_menu', function() {
    add_options_page('Настройка времени доставки', 'Настройка времени доставки', 'manage_options', 'wc-delivery-times-settings', 'wcdtwc_settings_page_html');
});

function wcdtwc_settings_page_html() {
    if (!current_user_can('manage_options')) return;

    if (isset($_POST['wcdtwc_submit'])) {
        check_admin_referer('wcdtwc_save_settings');
        $working_hours = sanitize_text_field($_POST['working_hours']);
        $preparation_time = intval($_POST['preparation_time']);
        $required = $_POST['delivery_datetime_required'] === 'yes' ? 'yes' : 'no';
        update_option('wcdtwc_delivery_times_settings', [
            'working_hours' => $working_hours,
            'preparation_time' => $preparation_time,
            'delivery_datetime_required' => $required
        ]);
        echo '<div class="updated"><p>Настройки сохранены</p></div>';
    }

    $options = get_option('wcdtwc_delivery_times_settings', [
        'working_hours' => '10:00-22:00',
        'preparation_time' => 30,
        'delivery_datetime_required' => 'no'
    ]);
    ?>
    <div class="wrap">
        <h1>Настройки доставки</h1>
        <form method="post" action="">
            <?php wp_nonce_field('wcdtwc_save_settings'); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row"><label for="working_hours">Рабочие часы (например: 10:00-22:00)</label></th>
                    <td><input name="working_hours" type="text" id="working_hours" value="<?php echo esc_attr($options['working_hours']); ?>" class="regular-text"></td>
                </tr>
                <tr>
                    <th scope="row"><label for="preparation_time">Время приготовления (в минутах)</label></th>
                    <td><input name="preparation_time" type="number" id="preparation_time" value="<?php echo esc_attr($options['preparation_time']); ?>" min="0" class="small-text"></td>
                </tr>
                <tr>
                    <th scope="row">Обязательный выбор даты и времени доставки</th>
                    <td>
                        <label><input type="radio" name="delivery_datetime_required" value="yes" <?php checked($options['delivery_datetime_required'], 'yes'); ?>> Да</label><br>
                        <label><input type="radio" name="delivery_datetime_required" value="no" <?php checked($options['delivery_datetime_required'], 'no'); ?>> Нет</label>
                    </td>
                </tr>
            </table>
            <?php submit_button('Сохранить настройки', 'primary', 'wcdtwc_submit'); ?>
        </form>
    </div>
    <?php
}

// Название директории и название основного файла плагина Woocommerce
$required_plugin = 'woocommerce/woocommerce.php';

// Проверяем активность плагина
if ( is_plugin_active($required_plugin) ) {
    // Если плагин установлен и активирован, продолжаем выполнение нашего плагина
    add_action('plugins_loaded', 'my_plugin_init');
    
    function my_plugin_init() {
        // Логика активации вашего плагина
        echo "Ваш плагин успешно активировался!";
    }
} else {
    // Сообщение об ошибке при отсутствии активного плагина
    deactivate_plugins(plugin_basename(__FILE__)); // Деактивируем свой плагин
    wp_die("Ошибка: ваш плагин требует активной версии плагина Woocommerce.");
}