<?php

/**
 * Файл, определяющий основной класс плагина
 *
 * Класс, включающий атрибуты и функции, используемые как в публичной части сайта, так и в панели администратора.
 */

/**
 * Основной класс плагина.
 *
 * Используется для определения интернационализации, хуков администратора и хуков публичной части сайта.
 *
 * Также содержит уникальный идентификатор плагина и его текущую версию.
 */
class DTWC {

	/**
	 * Загрузчик, отвечающий за управление и регистрацию всех хуков, обеспечивающих работу плагина.
	 *
	 * @access   protected
	 * @var      Delivery_Times_For_WooCommerce_Loader    $loader    Управляет и регистрирует все хуки плагина.
	 */
	protected $loader;

	/**
	 * Уникальный идентификатор плагина.
	 *
	 * @access   protected
	 * @var      string    $plugin_name    Уникальный идентификатор плагина.
	 */
	protected $plugin_name;

	/**
	 * Текущая версия плагина.
	 *
	 * @access   protected
	 * @var      string    $version    Текущая версия плагина.
	 */
	protected $version;

	/**
	 * Определение основной функциональности плагина.
	 *
	 * Устанавливает название и версию плагина, загружает зависимости, задает локаль и регистрирует хуки для панели администратора и публичной части сайта.
	 */
	public function __construct() {
		$this->plugin_name = 'delivery-times-wc';
		$this->version     = '1.0';

		if ( defined( 'DTWC_VERSION' ) ) {
			$this->version = DTWC_VERSION;
		}

		$this->load_dependencies();
		$this->define_admin_hooks();
		$this->define_public_hooks();
	}

	/**
	 * Загрузка необходимых зависимостей для плагина.
	 *
	 * @access   private
	 */
	private function load_dependencies() {
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-dtwc-loader.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-dtwc-admin.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-dtwc-admin-settings.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/dtwc-admin-settings.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/dtwc-helper-functions.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/dtwc-woocommerce-checkout.php';
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/dtwc-woocommerce-settings.php';
		if ( function_exists( 'run_ddwc' ) ) {
			require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/dtwc-ddwc-settings.php';
		}
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-dtwc-public.php';
		$this->loader = new Delivery_Times_For_WooCommerce_Loader();
	}

	/**
	 * Регистрация хуков для функциональности панели администратора.
	 *
	 * @access   private
	 */
	private function define_admin_hooks() {
		$plugin_admin = new Delivery_Times_For_WooCommerce_Admin( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
	}

	/**
	 * Регистрация хуков для функциональности публичной части сайта.
	 *
	 * @access   private
	 */
	private function define_public_hooks() {
		$plugin_public = new Delivery_Times_For_WooCommerce_Public( $this->get_plugin_name(), $this->get_version() );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
	}

	/*Запуск загрузчика для выполнения всех хуков в WordPress*/
	public function run() {
		$this->loader->run();
	}

	/**
	 * Получение названия плагина.
	 * @return   string    Название плагина.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Получение загрузчика плагина.
	 * @return   Delivery_Times_For_WooCommerce_Loader    Объект загрузчика хуков плагина.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Получение версии плагина.
	 *
	 * @return   string    Версия плагина.
	 */
	public function get_version() {
		return $this->version;
	}
}