<?php
/**
 * Регистрирует все действия и фильтры для плагина.
 *
 * Поддерживает список всех хуков, зарегистрированных в плагине, и регистрирует их в API WordPress.
 * Вызывает функцию run для выполнения списка действий и фильтров.
 *
 */
class Delivery_Times_For_WooCommerce_Loader {

	/**
	 * Массив действий, зарегистрированных в WordPress.
	 *
	 * @access   protected
	 * @var      array    $actions    Действия, зарегистрированные в WordPress, выполняемые при загрузке плагина.
	 */
	protected $actions;

	/**
	 * Массив фильтров, зарегистрированных в WordPress.
	 *
	 * @access   protected
	 * @var      array    $filters    Фильтры, зарегистрированные в WordPress, выполняемые при загрузке плагина.
	 */
	protected $filters;

	/**
	 * Инициализация коллекций для хранения действий и фильтров.
	 */
	public function __construct() {
		$this->actions = array();
		$this->filters = array();
	}

	/**
	 * Добавляет новое действие в коллекцию для регистрации в WordPress.
	 *
	 * @param    string    $hook             Название действия WordPress, которое регистрируется.
	 * @param    object    $component        Ссылка на объект, в котором определено действие.
	 * @param    string    $callback         Название функции в $component.
	 * @param    int       $priority         (Опционально) Приоритет выполнения. По умолчанию 10.
	 * @param    int       $accepted_args    (Опционально) Количество аргументов, передаваемых в $callback. По умолчанию 1.
	 */
	public function add_action( $hook, $component, $callback, $priority = 10, $accepted_args = 1 ) {
		$this->actions = $this->add( $this->actions, $hook, $component, $callback, $priority, $accepted_args );
	}

	/**
	 * Добавляет новый фильтр в коллекцию для регистрации в WordPress.
	 *
	 * @param    string    $hook             Название фильтра WordPress, который регистрируется.
	 * @param    object    $component        Ссылка на объект, в котором определен фильтр.
	 * @param    string    $callback         Название функции в $component.
	 * @param    int       $priority         (Опционально) Приоритет выполнения. По умолчанию 10.
	 * @param    int       $accepted_args    (Опционально) Количество аргументов, передаваемых в $callback. По умолчанию 1.
	 */
	// public function add_filter( $hook, $component, $callback, $priority = 10, $accepted_args = 1 ) {
		// $this->filters = $this->add( $this->filters, $hook, $component, $callback, $priority, $accepted_args );
	// }

	/**
	 * Вспомогательная функция для регистрации действий и фильтров в одной коллекции.
	 *
	 * @access   private
	 * @param    array     $hooks            Коллекция зарегистрированных хуков (действия или фильтры).
	 * @param    string    $hook             Название действия или фильтра WordPress.
	 * @param    object    $component        Ссылка на объект, в котором определен хук.
	 * @param    string    $callback         Название функции в $component.
	 * @param    int       $priority         Приоритет выполнения.
	 * @param    int       $accepted_args    Количество аргументов, передаваемых в $callback.
	 * @return   array                       Коллекция зарегистрированных хуков в WordPress.
	 */
	private function add( $hooks, $hook, $component, $callback, $priority, $accepted_args ) {
		$hooks[] = array(
			'hook'          => $hook,
			'component'     => $component,
			'callback'      => $callback,
			'priority'      => $priority,
			'accepted_args' => $accepted_args
		);
		return $hooks;
	}

	/**
	 * Регистрирует фильтры и действия в WordPress.
	 */
	public function run() {
		foreach ( $this->filters as $hook ) {
			add_filter( $hook['hook'], array( $hook['component'], $hook['callback'] ), $hook['priority'], $hook['accepted_args'] );
		}
		foreach ( $this->actions as $hook ) {
			add_action( $hook['hook'], array( $hook['component'], $hook['callback'] ), $hook['priority'], $hook['accepted_args'] );
		}
	}
}